#!/usr/bin/env sh

# Only accepts 1 argument: the Minecraft version to download
LOOOKING_FOR=
if [ -z "${1}" ]; then
	echo "ERROR: Requires one argument (the Minecraft instance to launch)"
	exit 1
else
	LOOKING_FOR="${1}"
fi

PLATFORM=$(uname -s | tr '[:upper:]' '[:lower:]')

MANIFEST_LIST_URL='https://launchermeta.mojang.com/mc/game/version_manifest.json'

mkdir -p assets/indexes

echo 'Retrieving version manifest...'
VERSION_MANIFEST_URL=$(curl "${MANIFEST_LIST_URL}" 2>/dev/null | jq -r "(.versions.[] | select(.id==\"${LOOKING_FOR}\")).url")

if [ -z "${VERSION_MANIFEST_URL}" ]; then
	echo "ERROR: Unable to find requested version"
	exit 1
fi

mkdir -p manifests
VERSION_MANIFEST=$(curl ${VERSION_MANIFEST_URL} 2>/dev/null | sed -e 's/\\/\\\\/g' | tee "manifests/${LOOKING_FOR}.json") # Some versions contain illegal escape sequences which will crash some shells such as Ksh; double-escape it just to be safe

# 1.12 and earlier, natives and regular libraries are contained in different elements. For 1.13 and later, an element may contain both.
echo 'Downloading libraries...'
for I in $(echo "${VERSION_MANIFEST}" | jq -c .libraries.[]); do
	if [[ $(echo "${I}" | jq .downloads.artifact) != 'null' ]]; then
		echo "Library '$(echo ${I} | jq -r .name )' found - downloading..."
		mkdir -p libraries/$(dirname $(echo ${I} | jq -r .downloads.artifact.path))
		curl $(echo ${I} | jq -r .downloads.artifact.url) > libraries/$(echo "${I}" | jq -r .downloads.artifact.path) 2>/dev/null
	fi
	if [[ $(echo "${I}" | jq .natives) != 'null' ]]; then
		echo "Downloading native for platform '${PLATFORM}'..."
		if [[ $(echo ${I} | jq -r ".downloads.classifiers.\"natives-${PLATFORM}\"") == 'null' ]]; then
			echo "WARNING: Mojang does not distribute a '$(echo ${I} | jq -r .name)' native for this platform. This is probably not an issue unless you are using an obscure platform."
		else
			mkdir -p libraries/$(dirname $(echo ${I} | jq -r ".downloads.classifiers.\"natives-${PLATFORM}\".path"))
			curl $(echo ${I} | jq -r ".downloads.classifiers.\"natives-${PLATFORM}\".url") > libraries/$(echo "${I}" | jq -r ".downloads.classifiers.\"natives-${PLATFORM}\".path") 2>/dev/null
		fi
	fi
done

echo 'Retrieving asset index...'
ASSET_INDEX=$(echo "${VERSION_MANIFEST}" | jq .assetIndex)

echo 'Retrieving object list...'
OBJECT_LIST=$(curl $(echo ${ASSET_INDEX} | jq -r .url) 2>/dev/null | tee assets/indexes/$(echo ${ASSET_INDEX} | jq -r .id).json)

HASH_COUNT=$(echo ${OBJECT_LIST} | jq '.objects | length')

echo 'Downloading required assets. This may take a few minutes.'
MCRESOURCES='https://resources.download.minecraft.net/'
if [[ "$(echo "${ASSET_INDEX}" | jq -r .id)" == 'pre-1.6' || "$(echo "${ASSET_INDEX}" | jq -r .id)" == 'legacy' ]]; then # Release 1.6 and prior use a different asset layout
	while read -r LINE; do
		H=$(echo ${OBJECT_LIST} | jq -r ".objects.[\"${LINE}\"].hash")
 		DIR="${H[0]}${H[1]}/"

		mkdir -p "$(dirname "assets/virtual/legacy/${LINE}")"
		curl "${MCRESOURCES}${DIR}${H}" > "assets/virtual/legacy/${LINE}" 2>/dev/null
		echo "Downloaded: assets/virtual/legacy/${LINE}"
	done <<< "$(echo "${OBJECT_LIST}" | jq -r '.objects | keys[]')"
else
	IND=0
	for I in $(echo "${OBJECT_LIST}" | jq -r .objects.[].hash); do
		IND=$(echo "${IND}" + 1 | nickle)
		DIR="$(echo "${I}" | cut -c1-2)/"

		mkdir -p "assets/objects/${DIR}"

		# If file exists, check its integrity with hash and skip if it's good
		if [[ -f assets/objects/${DIR}${I} && $(cat "assets/objects/${DIR}${I}" | sha1sum | sed -e 's/  (stdin)//') == "${I}" ]]; then # Removing (stdin) is not needed for most sha1sum implementations
				echo "${IND}/${HASH_COUNT} (${DIR}${I}) is already downloaded; skipping. ($(echo "printf(\"%5l\", ${IND} / ${HASH_COUNT} * 100)" | nickle)%)"
			continue;
		fi

		curl "${MCRESOURCES}${DIR}${I}" > "assets/objects/${DIR}${I}" 2>/dev/null
		echo "Downloading ${IND}/${HASH_COUNT}: ${MCRESOURCES}${DIR}${I} ($(echo "printf(\"%5l\", ${IND} / ${HASH_COUNT} * 100)" | nickle)%)"
	done
fi

echo 'Downloading minecraft jar...'
CLIENT_URL=$(echo "${VERSION_MANIFEST}" | jq -r .downloads.client.url)
mkdir -p "libraries/com/mojang/minecraft/${LOOKING_FOR}"
curl "${CLIENT_URL}" > "libraries/com/mojang/minecraft/${LOOKING_FOR}/minecraft-${LOOKING_FOR}-client.jar" 2>/dev/null

echo 'Finished!'
