#!/usr/bin/env sh

PLATFORM=$(uname -s | tr '[:upper:]' '[:lower:]')

# Only accepts 1 argument: the Minecraft profile to launch
INAME=
if [ -z "${1}" ]; then
	echo "ERROR: Requires one argument (the Minecraft instance to launch)"
	exit 1
else
	INAME="${1}"
fi

# Setup and read configurations
JAVA=java
MCUSER=Player
JARGS='-Xms512M -Xmx1024M'

VERSION=

BLACKLISTED_LIBRARIES=
BLACKLISTED_NATIVES=

ADDITIONAL_LIBRARIES=

XARGS=

if [ -f 'user.local' ]; then
	. "./user.local"
else
	echo "JAVA=${JAVA}" > user.local
	echo "MCUSER=${MCUSER}" >> user.local
	echo "JARGS='${JARGS}'" >> user.local
fi

if [ -f "profiles.local/${INAME}" ]; then
	. "profiles.local/${INAME}"
elif [ -f "profiles/${INAME}" ]; then
	. "profiles/${INAME}"
else
	echo "ERROR: Specified profile does not exist."
	exit 1
fi

if [ ! -f "manifests/${VERSION}.json" ]; then
	echo "Invalid version detected. Either it is mistyped, or you have not downloaded it yet."
	exit 1
fi

MINECRAFT_JAR=${MINECRAFT_JAR:=libraries/com/mojang/minecraft/${VERSION}/minecraft-${VERSION}-client.jar}

VERSION_MANIFEST=$(cat "manifests/${VERSION}.json")

# Some libraries included in the manifests are known to cause issues, so we blacklist them here
# 1.12 and earlier, natives and regular libraries are contained in different elements. For 1.13 and later, an element may contain both.
CLASSPATH=
for I in $(echo "${VERSION_MANIFEST}" | jq -c .libraries.[]); do
	if [[ $(echo "${I}" | jq .downloads.artifact) != 'null' ]]; then
		for J in ${BLACKLISTED_LIBRARIES}; do
			if [ ${J} == $(basename $(echo ${I} | jq -r .downloads.artifact.path)) ]; then
				SKIP=1
			fi
		done
		if [[ ${SKIP} -eq 1 ]]; then
			SKIP=
			continue
		fi
		JAR=$(echo ${I} | jq -r .downloads.artifact.path)
		if [ "${JAR}" != 'null' ]; then
			CLASSPATH="${PWD}/libraries/${JAR}:${CLASSPATH}"
		fi
	fi
	if [[ $(echo "${I}" | jq .natives) != 'null' ]]; then
		for J in ${BLACKLISTED_NATIVES}; do
			if [ ${J} == $(basename $(echo ${I} | jq ".downloads.classifiers.\"natives-${PLATFORM}\".path")) ]; then
				SKIP=1
			fi
		done
		if [[ ${SKIP} -eq 1 ]]; then
			SKIP=
			continue
		fi
		JAR=$(echo ${I} | jq -r ".downloads.classifiers.\"natives-${PLATFORM}\".path")
		if [ "${JAR}" != 'null' ]; then
			CLASSPATH="${PWD}/libraries/${JAR}:${CLASSPATH}"
		fi
	fi
done

for I in ${ADDITIONAL_LIBRARIES}; do
	CLASSPATH="${CLASSPATH}:${PWD}/${I}"
done

CLASSPATH="${CLASSPATH}:${PWD}/${MINECRAFT_JAR}"

if [[ $(echo "${VERSION_MANIFEST}" | jq -r .minecraftArguments) != 'null' ]]; then
	ARGUMENT_LIST=$(echo ${VERSION_MANIFEST} | jq -r .minecraftArguments)
else
	for I in $(echo ${VERSION_MANIFEST} | jq -r '.arguments.game[] | select(type=="string")'); do
		ARGUMENT_LIST="${ARGUMENT_LIST} ${I}"
	done
fi

ARGUMENTS="$(echo ${ARGUMENT_LIST} | sed -e "s/\${auth_player_name}/${MCUSER}/" -e "s/\${version_name}/${VERSION}/" -e "s@\${game_directory}@\"${PWD}/instances/${INAME}\"@" -e "s@\${assets_root}@${PWD}/assets@" -e "s/\${assets_index_name}/$(echo ${VERSION_MANIFEST} | jq -r .assetIndex.id)/" -e 's/${auth_uuid}/00000000-0000-0000-0000-000000000000/' -e 's/${auth_access_token}/0/' -e 's/${user_type}/mojang/' -e "s/\${version_type}/$(echo ${VERSION_MANIFEST} | jq -r .type)/" -e 's/${user_properties}/{}/' -e "s@\${game_assets}@${PWD}/assets/virtual/legacy@" -e 's/${auth_session}/0/')"

MAINCLASS=${MAINCLASS:=$(echo ${VERSION_MANIFEST} | jq -r .mainClass)}

INSTANCES="${PWD}/instances"
if [ -f "playtime/${INAME}" ]; then
	PRVTOTAL=`cat "playtime/${INAME}"`
else
	PRVTOTAL=0
fi
STARTTIME=`date +"%s"`
mkdir -p "${INSTANCES}/${INAME}"
ORIGDIR="${PWD}"
cd "${INSTANCES}/${INAME}"
trap 'echo "Interrupt received"' INT

echo ${ARGUMENTS} ${XARGS} | xargs "${JAVA}" ${JARGS} -Duser.language=en -Djava.library.path="${ORIGDIR}/natives-${PLATFORM}" -cp "${CLASSPATH}" "${MAINCLASS}"

ENDTIME=`date +"%s"`
cd "${ORIGDIR}"

TOTALSECONDS=`echo ${ENDTIME} - ${STARTTIME} + ${PRVTOTAL} | nickle`
if [ ! -z ${TOTALSECONDS} ]; then # Prevent playtime from being cleared if nickle is broken/missing
	mkdir -p playtime
	echo ${TOTALSECONDS} > "playtime/${INAME}"
	echo Total playtime: ${ENDTIME} - ${STARTTIME} + ${PRVTOTAL} = ${TOTALSECONDS}
fi
